/*
 * Decompiled with CFR 0.152.
 */
package mobac.program.download;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.HttpURLConnection;
import mobac.exceptions.DownloadFailedException;
import mobac.exceptions.UnrecoverableDownloadException;
import mobac.program.model.Settings;
import mobac.program.tilestore.TileStore;
import mobac.program.tilestore.TileStoreEntry;
import mobac.utilities.Utilities;
import mobac.utilities.tar.TarIndexedArchive;
import org.apache.log4j.Logger;
import org.openstreetmap.gui.jmapviewer.interfaces.MapSource;
import org.openstreetmap.gui.jmapviewer.interfaces.MapSpace;

public class TileDownLoader {
    public static String ACCEPT = "text/html, image/png, image/jpeg, image/gif, */*;q=0.1";
    private static Logger log;
    private static Settings settings;

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static int getImage(int layer, int x, int y, int zoom, MapSource mapSource, TarIndexedArchive tileArchive) throws IOException, InterruptedException, UnrecoverableDownloadException {
        MapSpace mapSpace = mapSource.getMapSpace();
        int maxTileIndex = mapSpace.getMaxPixels(zoom) / mapSpace.getTileSize();
        if (x > maxTileIndex) {
            throw new RuntimeException("Invalid tile index x=" + x + " for zoom " + zoom);
        }
        if (y > maxTileIndex) {
            throw new RuntimeException("Invalid tile index y=" + y + " for zoom " + zoom);
        }
        TileStore ts = TileStore.getInstance();
        Settings s = Settings.getInstance();
        String tileFileName = String.format("l%dx%dy%d", layer, x, y);
        TileStoreEntry tile = null;
        if (s.tileStoreEnabled) {
            tile = ts.getTile(x, y, zoom, mapSource);
            boolean expired = TileDownLoader.isTileExpired(tile);
            if (tile != null) {
                if (expired) {
                    log.trace("Expired: " + mapSource.getName() + " " + tile);
                } else {
                    TarIndexedArchive tarIndexedArchive = tileArchive;
                    synchronized (tarIndexedArchive) {
                        log.trace("Tile used from tilestore");
                        tileArchive.writeFileFromData(tileFileName, tile.getData());
                    }
                    return 0;
                }
            }
        }
        byte[] data = null;
        if (tile == null) {
            data = TileDownLoader.downloadTileAndUpdateStore(x, y, zoom, mapSource);
        } else {
            TileDownLoader.updateStoredTile(tile, mapSource);
            data = tile.getData();
        }
        if (data == null) {
            return 0;
        }
        TarIndexedArchive tarIndexedArchive = tileArchive;
        synchronized (tarIndexedArchive) {
            tileArchive.writeFileFromData(tileFileName, data);
        }
        return data.length;
    }

    public static byte[] downloadTileAndUpdateStore(int x, int y, int zoom, MapSource mapSource) throws UnrecoverableDownloadException, IOException, InterruptedException {
        HttpURLConnection conn = mapSource.getTileUrlConnection(zoom, x, y);
        if (conn == null) {
            throw new UnrecoverableDownloadException("Tile x=" + x + " y=" + y + " zoom=" + zoom + " is not a valid tile in map source " + mapSource);
        }
        log.trace("Downloading " + conn.getURL());
        conn.setRequestMethod("GET");
        Settings s = Settings.getInstance();
        conn.setConnectTimeout(1000 * s.httpConnectionTimeout);
        conn.setReadTimeout(1000 * s.httpReadTimeout);
        conn.addRequestProperty("User-agent", s.getUserAgent());
        conn.setRequestProperty("Accept", ACCEPT);
        conn.connect();
        int code = conn.getResponseCode();
        if (code != 200) {
            throw new DownloadFailedException(code);
        }
        String contentType = conn.getContentType();
        if (contentType != null && !contentType.startsWith("image/")) {
            throw new UnrecoverableDownloadException("Content type of the loaded image is unknown: " + contentType);
        }
        String eTag = conn.getHeaderField("ETag");
        long timeLastModified = conn.getLastModified();
        long timeExpires = conn.getExpiration();
        byte[] data = TileDownLoader.loadTileInBuffer(conn);
        Utilities.checkForInterruption();
        String imageFormat = Utilities.getImageDataFormat(data);
        if (imageFormat == null) {
            throw new UnrecoverableDownloadException("The returned image is of unknown format");
        }
        if (mapSource.allowFileStore() && s.tileStoreEnabled) {
            TileStore.getInstance().putTileData(data, x, y, zoom, mapSource, timeLastModified, timeExpires, eTag);
        }
        Utilities.checkForInterruption();
        return data;
    }

    public static byte[] updateStoredTile(TileStoreEntry tile, MapSource mapSource) throws UnrecoverableDownloadException, IOException, InterruptedException {
        int x = tile.getX();
        int y = tile.getY();
        int zoom = tile.getZoom();
        MapSource.TileUpdate tileUpdate = mapSource.getTileUpdate();
        switch (tileUpdate) {
            case ETag: {
                boolean unchanged = TileDownLoader.hasTileETag(tile, mapSource);
                if (!unchanged) break;
                if (log.isTraceEnabled()) {
                    log.trace("Data unchanged on server (eTag): " + mapSource + " " + tile);
                }
                return null;
            }
            case LastModified: {
                boolean isNewer = TileDownLoader.isTileNewer(tile, mapSource);
                if (isNewer) break;
                if (log.isTraceEnabled()) {
                    log.trace("Data unchanged on server (LastModified): " + mapSource + " " + tile);
                }
                return null;
            }
        }
        HttpURLConnection conn = mapSource.getTileUrlConnection(zoom, x, y);
        if (conn == null) {
            throw new UnrecoverableDownloadException("Tile x=" + x + " y=" + y + " zoom=" + zoom + " is not a valid tile in map source " + mapSource);
        }
        if (log.isTraceEnabled()) {
            log.trace(String.format("Checking %s %s", mapSource.getName(), tile));
        }
        conn.setRequestMethod("GET");
        boolean conditionalRequest = false;
        switch (tileUpdate) {
            case IfNoneMatch: {
                if (tile.geteTag() == null) break;
                conn.setRequestProperty("If-None-Match", tile.geteTag());
                conditionalRequest = true;
                break;
            }
            case IfModifiedSince: {
                if (tile.getTimeLastModified() <= 0L) break;
                conn.setIfModifiedSince(tile.getTimeLastModified());
                conditionalRequest = true;
            }
        }
        Settings s = Settings.getInstance();
        conn.setConnectTimeout(1000 * s.httpConnectionTimeout);
        conn.setReadTimeout(1000 * s.httpReadTimeout);
        conn.addRequestProperty("User-agent", s.getUserAgent());
        conn.setRequestProperty("Accept", ACCEPT);
        conn.connect();
        int code = conn.getResponseCode();
        if (conditionalRequest && code == 304) {
            if (mapSource.allowFileStore() && s.tileStoreEnabled) {
                tile.update(conn.getExpiration());
                TileStore.getInstance().putTile(tile, mapSource);
            }
            if (log.isTraceEnabled()) {
                log.trace("Data unchanged on server: " + mapSource + " " + tile);
            }
            return null;
        }
        if (code != 200) {
            throw new DownloadFailedException(code);
        }
        String contentType = conn.getContentType();
        if (contentType != null && !contentType.startsWith("image/")) {
            throw new UnrecoverableDownloadException("Content type of the loaded image is unknown: " + contentType);
        }
        String eTag = conn.getHeaderField("ETag");
        long timeLastModified = conn.getLastModified();
        long timeExpires = conn.getExpiration();
        byte[] data = TileDownLoader.loadTileInBuffer(conn);
        Utilities.checkForInterruption();
        String imageFormat = Utilities.getImageDataFormat(data);
        if (imageFormat == null) {
            throw new UnrecoverableDownloadException("The returned image is of unknown format");
        }
        if (mapSource.allowFileStore() && s.tileStoreEnabled) {
            TileStore.getInstance().putTileData(data, x, y, zoom, mapSource, timeLastModified, timeExpires, eTag);
        }
        Utilities.checkForInterruption();
        return data;
    }

    public static boolean isTileExpired(TileStoreEntry tileStoreEntry) {
        if (tileStoreEntry == null) {
            return true;
        }
        long expiredTime = tileStoreEntry.getTimeExpires();
        if (expiredTime >= 0L) {
            long maxExpirationTime = TileDownLoader.settings.tileMaxExpirationTime + tileStoreEntry.getTimeDownloaded();
            long minExpirationTime = TileDownLoader.settings.tileMinExpirationTime + tileStoreEntry.getTimeDownloaded();
            expiredTime = Math.max(minExpirationTime, Math.min(maxExpirationTime, expiredTime));
        } else {
            expiredTime = tileStoreEntry.getTimeDownloaded() + TileDownLoader.settings.tileDefaultExpirationTime;
        }
        return expiredTime < System.currentTimeMillis();
    }

    protected static byte[] loadTileInBuffer(HttpURLConnection conn) throws IOException {
        InputStream input = conn.getInputStream();
        int bufSize = Math.max(input.available(), 32768);
        ByteArrayOutputStream bout = new ByteArrayOutputStream(bufSize);
        byte[] buffer = new byte[2048];
        boolean finished = false;
        do {
            int read;
            if ((read = input.read(buffer)) >= 0) {
                bout.write(buffer, 0, read);
                continue;
            }
            finished = true;
        } while (!finished);
        if (bout.size() == 0) {
            return null;
        }
        return bout.toByteArray();
    }

    protected static boolean isTileNewer(TileStoreEntry tile, MapSource mapSource) throws IOException {
        long oldLastModified = tile.getTimeLastModified();
        if (oldLastModified <= 0L) {
            log.warn("Tile age comparison not possible: tile in tilestore does not contain lastModified attribute");
            return true;
        }
        HttpURLConnection conn = mapSource.getTileUrlConnection(tile.getZoom(), tile.getX(), tile.getY());
        conn.setRequestMethod("HEAD");
        conn.setRequestProperty("Accept", ACCEPT);
        long newLastModified = conn.getLastModified();
        if (newLastModified == 0L) {
            return true;
        }
        return newLastModified > oldLastModified;
    }

    protected static boolean hasTileETag(TileStoreEntry tile, MapSource mapSource) throws IOException {
        String eTag = tile.geteTag();
        if (eTag == null || eTag.length() == 0) {
            log.warn("ETag check not possible: tile in tilestore does not contain ETag attribute");
            return true;
        }
        HttpURLConnection conn = mapSource.getTileUrlConnection(tile.getZoom(), tile.getX(), tile.getY());
        conn.setRequestMethod("HEAD");
        conn.setRequestProperty("Accept", ACCEPT);
        String onlineETag = conn.getHeaderField("ETag");
        if (onlineETag == null || onlineETag.length() == 0) {
            return true;
        }
        return onlineETag.equals(eTag);
    }

    static {
        System.setProperty("sun.net.client.defaultReadTimeout", "15000");
        System.setProperty("http.maxConnections", "20");
        log = Logger.getLogger(TileDownLoader.class);
        settings = Settings.getInstance();
    }
}

